<?php
// cron.php - این اسکریپت باید توسط سرور به صورت خودکار اجرا شود

// جلوگیری از اجرای این فایل از طریق مرورگر برای امنیت
if (php_sapi_name() !== 'cli') {
    die("Access Denied. This script can only be run from the command line.");
}

// برای اجرای صحیح، به ریشه پروژه میرویم
chdir(__DIR__);

require_once 'core/config.php';
require_once 'core/database.php';

echo "Cron Job Started: " . date('Y-m-d H:i:s') . "\n";

try {
    // --- بخش اول: منقضی کردن سرویس‌های تاریخ گذشته ---
    $stmt_expire = $pdo->prepare(
        "UPDATE orders SET status = 'expired' WHERE expiry_date < NOW() AND status = 'active'"
    );
    
    $stmt_expire->execute();
    
    // دریافت تعداد ردیف‌هایی که آپدیت شدند
    $expired_count = $stmt_expire->rowCount();
    echo "- " . $expired_count . " services have been marked as expired.\n";

    // --- بخش دوم: ارسال یادآوری تمدید ---
    $reminder_days = (int)setting('renewal_reminder_days', 3);
    if ($reminder_days > 0) {
        // پیدا کردن سفارش‌هایی که در X روز آینده منقضی می‌شوند و هنوز یادآوری برایشان ارسال نشده
        $target_date = date('Y-m-d H:i:s', time() + ($reminder_days * 86400));
        
        $orders_to_remind_stmt = $pdo->prepare(
            "SELECT id, user_id, marzban_username, expiry_date 
             FROM orders 
             WHERE status = 'active' AND reminder_sent = 0 AND expiry_date BETWEEN NOW() AND ?"
        );
        $orders_to_remind_stmt->execute([$target_date]);
        $orders_to_remind = $orders_to_remind_stmt->fetchAll(PDO::FETCH_ASSOC);

        $notification_count = 0;
        if ($orders_to_remind) {
            $notification_stmt = $pdo->prepare(
                "INSERT INTO notifications (user_id, title, message, link) VALUES (?, ?, ?, ?)"
            );
            $update_order_stmt = $pdo->prepare("UPDATE orders SET reminder_sent = 1 WHERE id = ?");

            foreach ($orders_to_remind as $order) {
                $title = "یادآوری تمدید سرویس";
                $remaining_days = floor((strtotime($order['expiry_date']) - time()) / 86400);
                if ($remaining_days < 0) $remaining_days = 0; // جلوگیری از نمایش عدد منفی
                
                $message = "سرویس " . htmlspecialchars($order['marzban_username']) . " شما تا " . $remaining_days . " روز دیگر منقضی می‌شود. لطفاً برای جلوگیری از قطعی، آن را تمدید کنید.";
                $link = "user/my-services.php";
                
                // ارسال اعلان
                $notification_stmt->execute([$order['user_id'], $title, $message, $link]);
                
                // علامت‌گذاری سفارش به عنوان "یادآوری ارسال شده" برای جلوگیری از ارسال مجدد
                $update_order_stmt->execute([$order['id']]);
                
                $notification_count++;
            }
        }
        echo "- " . $notification_count . " renewal reminders have been sent.\n";
    }

} catch (PDOException $e) {
    // در صورت بروز خطا، آن را در لاگ سرور ثبت می‌کنیم
    error_log("Cron Job Failed: " . $e->getMessage());
    echo "An error occurred: " . $e->getMessage() . "\n";
}

echo "Cron Job Finished: " . date('Y-m-d H:i:s') . "\n";